function classifiers = classify_cuda(tsp_struct)
%CLASSIFY_CUDA Takes the output of a run of TSP_CUDA or KTSP_CUDA and creates the classifier
%
%   [CLASSIFIERS] = TSP_CUDA(TSP_STRUCT) creates the classifier for TSP or kTSP
%   based on the primary and secondary scores in the tsp_struct and the size of 
%   the classifier (1 for TSP, determined by kTSP algorithm).  The classifier 
%   is returned in an array of structs, where each element contains information
%   on that portion of the classifier sorted by primary then secondary tsp score.
%
%   CLASSIFIERS is an array of structs, each containing the following fields:
%   indexi: 	index of the first probe from the original probe list
%   indexj: 	index of the second probe from the original probe list
%   primary: 	primary TSP score
%   secondary: 	secondary TSP score
%   name1: 	name of probe1
%   name2:  	name of probe2

% Check inputs
if (nargin < 1)
	error('Usage: [CLASSIFIERS] = CLASSIFY_CUDA(TSP_STRUCT)');
end

% Check to make sure tsp_struct is a struct
if ~isstruct(tsp_struct)
	error('Invalid input (should be tsp_struct format)');
end

% Get the k disjoint pairs
[sorted, secondary, indexi, indexj] = nvdisjointpairmex(tsp_struct.primary, tsp_struct.secondary, tsp_struct.k, 0);

% Check to make sure we have actually returned the correct number
if (size(sorted, 1) ~= tsp_struct.k)
	fprintf('Warning: optimal k is %d but only found %d disjoint pairs in final search\n', tsp_struct.k, size(sorted, 1));
	tsp_struct.k = size(sorted, 1);
end

% If we have filtered the data by differential expression, we want to return 
% indices and names relative to the original data set
classifiers = [];

for j=1:tsp_struct.k
	
	classifer = struct;
	if (tsp_struct.filter ~= 0)
		classifier.indexi = tsp_struct.indices(indexi(j));
		classifier.indexj = tsp_struct.indices(indexj(j));
		classifier.primary = sorted(j);
		classifier.secondary = secondary(j);
		classifier.name1 = tsp_struct.probes(tsp_struct.indices(indexi(j)));
		classifier.name2 = tsp_struct.probes(tsp_struct.indices(indexj(j)));
	else
		classifier.indexi = indexi(j);
		classifier.indexj = indexj(j);
		classifier.primary = sorted(j);
		classifier.secondary = secondary(j);
		classifier.name1 = tsp_struct.probes(indexi(j));
		classifier.name2 = tsp_struct.probes(indexj(j));
	end

	classifiers = [classifiers classifier];
end
	