function [error_rate, predictions] = cv_cuda(classifier, data, labels)
%CV_CUDA Takes a classifier of TSP and returns the error rate and predictions
%
%   [ERROR_RATE, PREDICTIONS] = CV_CUDA(CLASSIFIER, DATA) performs cross 
%   validation on the classifier using the provided data and class labels.  CLASSIFER
%   should be the output of the TSP_CLASSIFY algorithm.  DATA  should be the same as
%   used to create the classifier.  Labels are assumed to be the first row of DATA. 
%
%   [ERROR_RATE, PREDICTIONS] = CV_CUDA(CLASSIFIER, DATA, LABELS) performs cross 
%   validation on the classifier using the provided data and class labels.  CLASSIFER
%   should be the output of the TSP_CLASSIFY algorithm.  DATA and LABELS should
%   be the same as used to create the classifier. In this case, LABELS are explicitly
%   provided by the user and are NOT assumed to be the first row of DATA.
%    
%   Error rate is simply the percent correct predictions by this classifier for LOOCV. 
%   Predictions is a 2xN (where N is the number of samples) showing the prediction (first row)
%   and the actual class label (second row). 

	if (nargin < 3)
		% Get the labels from the data matrix first row
		labels = [];
	end
	if (nargin < 2)
		error('Usage: [ERROR_RATE, PREDICTIONS] = CV_CUDA(TSP_CLASSIFER, DATA, LABELS)');
	end

	% If the label set is empty, get the first row of the data matrix
	if (isempty(labels))
		labels = data(1,:);
		data(1,:) = [];	
	% Check to make sure the number of labels is ok
	elseif (length(labels) ~= size(data, 2))
		error('Number of class labels does not match number of cols of data');
	end

	% Check to make sure classifer is a struct
	if ~isstruct(classifier(1))
		error('Invalid input (should be tsp_classifier format)');
	end
	
	% Check to make sure this is not a TST struct
	if isfield(classifier, 'indexk')
		error('CV is not implemented for TST classifiers');
	end

	% Build the array of classifier indices from the array of structs
	indexi = [];
	indexj = [];
	for j=1:length(classifier)

		indexi = [indexi; classifier(j).indexi];
		indexj = [indexj; classifier(j).indexj];

	end

	[error_rate, predictions] = loocvmex(single(data), single(labels), int32([indexi indexj]));
				