/*=================================================================
 *
 *  LOOCV.h
 *  Author: Andrew Magis
 *  Do LOOCV of a classifier
 *  Inputs: data, labels, classifier (Mx2 matrix)
 *  Outputs: Error rate of classifier
 *
 *
 *=================================================================*/

 #ifndef _LOOCV_H
 #define _LOOCV_H
 
#include <math.h>
#include <vector>

//#define DEBUG 

float LOOCV(float *data, unsigned int m1, unsigned int n1, std::vector<float> labels, std::vector<ScoreElement> classifiers, int num_classifiers, int *indices, std::vector<int> preds) { 
		
	if (n1 != labels.size()) {
		throw GeneralException("Number of samples for data != number of labels\n", "LOOCV");
	}	

#ifdef DEBUG	
	printf("Data: [%d, %d]\n", m1, n1);
	printf("Performing LOOCV on classifier containing %d pairs\n", m3);
#endif
	
	//Count the numbers of each class
	float num_class1 = 0.f, num_class2 = 0.f;
	for (int i = 0; i < n1; i++) {
		if (labels[i] == 0.f)
			num_class1 += 1.f;
		else if (labels[i] == 1.f)
			num_class2 += 1.f;
		else 
			throw GeneralException("Invalid class label: must be all 0 or 1\n", "LOOCV");
	}
		
	//Num of correct predictions
	float num_correct = 0.f;
	//Num tried (less than n1 if there are ties)
	float num_tried = 0.f;
	
	float error = 0.f;
	
	//For each element of the dataset
	for (int sample = 0; sample < n1; sample++) {
	
		//Get the current class counts based on which type we are excluding
		float num_class1_current, num_class2_current;
		if (labels[sample] == 0.f) {
			num_class1_current = num_class1 - 1.f;
			num_class2_current = num_class2;
		} else if (labels[sample] == 1.f) {
			num_class1_current = num_class1;
			num_class2_current = num_class2 - 1.f;
		} else {
			throw GeneralException("Weird event: class labels must be all 0 or 1\n", "LOOCV");
		}
		
		//Vector to store the votes for each classifier
		std::vector<float> votes;
	
		//For each classifier
		for (int classifier = 0; classifier < num_classifiers; classifier++) {
			
			int indexi, indexj;
			if (indices == NULL) {
				indexi = classifiers[classifier].row;
				indexj = classifiers[classifier].col;
			} else {
				indexi = indices[classifiers[classifier].row];
				indexj = indices[classifiers[classifier].col];			
			}
			
			//Get pointers to the correct rows in the data
			float *data1 = &data[indexi];
			float *data2 = &data[indexj];
						
			float class1_score = 0.f, class2_score = 0.f;
			
			//For all samples (excluding sample above)
			for (int i = 0; i < n1; i++, data1+=m1, data2+=m1) {
						
				//Exclude the LOO sample
				if (i == sample) continue;
																	
				//Do the counts
				if (data1[0] < data2[0]) {
					if (labels[i] == 0.f)
						class1_score += 1.f;
					else
						class2_score += 1.f;
				}		
			}
			
			//At the end, get the probabilities using the adjusted class labels
			class1_score /= num_class1_current;
			class2_score /= num_class2_current;			
					
			//Choose the vote for this classifier based on these probabilities
			if (class1_score > class2_score) {
				votes.push_back(0.f);
			} else if (class2_score > class1_score) {
				votes.push_back(1.f);
			} else {
				votes.push_back(0.5f);
			}
		}
	
		
		/*
		//print votes
		printf("C Votes: ");
		for (int v = 0; v < votes.size(); v++) {
			printf("%.1f ", votes[v]);
		}
		printf("\n");
		*/	
			
		//Pointer to left out sample
		float *loo = data + m1*sample;
	
		//Vector of predictions
		std::vector<float> predictions;
	
		//Now we have collected all the votes for each classifier.  We must now
		//predict the left out sample using majority voting
		for (int classifier = 0; classifier < num_classifiers; classifier++) {	

			int indexi, indexj;
			if (indices == NULL) {
				indexi = classifiers[classifier].row;
				indexj = classifiers[classifier].col;
			} else {
				indexi = indices[classifiers[classifier].row];
				indexj = indices[classifiers[classifier].col];			
			}
			
			if (loo[indexi] < loo[indexj]) {
			
				//If this vote is 0
				if (votes[classifier] == 1.f) {
					predictions.push_back(1.f);
				} else if (votes[classifier] == 0.f) {
					predictions.push_back(0.f);
				} else {
					predictions.push_back(0.5f);
				}
			
			} else if (loo[indexi] > loo[indexj]) {
			
				//If this vote is 1
				if (votes[classifier] == 1.f) {
					predictions.push_back(0.f);
				} else if (votes[classifier] == 0.f) {
					predictions.push_back(1.f);
				} else {
					predictions.push_back(0.5f);
				}			
			
			} else {
				//The two are a tie, so no prediction possible
				predictions.push_back(0.5f);
			
			}
		}		
		
		//Finally, get the mean of the predictions.
		float mean = 0.f;
		for (int i = 0; i < predictions.size(); i++) {
			mean += predictions[i];
		}	
		mean /= (float)predictions.size();
		
		float prediction;
		if (mean < 0.5) {
			prediction = 0.f;
			num_tried += 1.f;
		} else if (mean > 0.5) {
			prediction = 1.f;
			num_tried += 1.f;
		} else {
			prediction = 0.5f;
		}
		
		//Finally check the prediction against the label
		if (prediction == labels[sample]) {
			num_correct += 1.f;
		}
		
		//Put the prediction and class label into the output array
		preds.push_back((int)prediction);
		
	}

	//Assign the error rate
	error = num_correct / num_tried;
	return error;
	
}

#endif
